<?php

/**
 * Retrieve information about installed plugins.
 *
 * @return array An associative array containing information about installed plugins.
 */
function get_installed_plugins() {
	$all_plugins       = get_plugins();
	$installed_plugins = [];

	if ( empty( $all_plugins ) ) {
		return [];
	}

	foreach ( $all_plugins as $plugin_path => $plugin_data ) {
		$main_file = basename( $plugin_path, '.php' );

		$installed_plugins[ $main_file ] = [
			'name'               => $plugin_data['Name'] ?? '',
			'version'            => $plugin_data['Version'] ?? '',
			'plugin_activated'   => is_plugin_active( $plugin_path ),
			'requires_wordpress' => $plugin_data['RequiresWP'] ?? '',
			'requires_php'       => $plugin_data['RequiresPHP'] ?? '',
		];
	}

	return $installed_plugins;
}

/**
 * Retrieve information about installed themes.
 *
 * @return array An associative array containing information about installed themes.
 */
function get_installed_themes() {
	$themes_directories = glob( ABSPATH . '/wp-content/themes/' . '*', GLOB_ONLYDIR );
	$installed_themes   = [];

	if ( empty( $themes_directories ) ) {
		return [];
	}

	foreach ( $themes_directories as $theme_directory ) {
		$theme_name     = basename( $theme_directory );
		$theme_data     = wp_get_theme( $theme_name );
		$style_css_path = $theme_directory . '/style.css';

		if ( ! file_exists( $style_css_path ) ) {
			continue;
		}

		$installed_themes[ $theme_name ] = [
			'name'               => $theme_data->get( 'Name' ),
			'version'            => $theme_data->get( 'Version' ),
			'theme_activated'    => get_option( 'stylesheet' ) === $theme_name,
			'requires_wordpress' => $theme_data->get( 'RequiresWP' ) ?? '',
			'requires_php'       => $theme_data->get( 'RequiresPHP' ) ?? '',
		];

		if ( is_multisite() ) {
			$network_enabled_themes                             = get_site_option( 'allowedthemes' ) ?: [];
			$installed_themes[ $theme_name ]['network_enabled'] = array_key_exists( $theme_name, $network_enabled_themes );
		}
	}

	return $installed_themes;
}

/**
 * Retrieves administrators without two-factor authentication enabled.
 *
 * @return object|null An object representing the result or null if no administrators without MFA are found.
 */
function get_admins_without_mfa() {
	global $wpdb;

	$secrets_mfa_table = $wpdb->prefix . 'wfls_2fa_secrets';

	// Query to get user IDs for administrators only
	$admins_query = $wpdb->prepare(
		"
        SELECT user_id
        FROM {$wpdb->usermeta}
        WHERE meta_key = '{$wpdb->prefix}capabilities'
        AND meta_value LIKE %s
        ",
		'%' . $wpdb->esc_like( 'administrator' ) . '%'
	);

	// Main query to get administrators without MFA
	$admins_without_mfa_query = "
        SELECT user_login
        FROM {$wpdb->users}
        LEFT JOIN {$secrets_mfa_table} ON {$wpdb->users}.ID = {$secrets_mfa_table}.user_id
        WHERE {$secrets_mfa_table}.user_id IS NULL
        AND {$wpdb->users}.ID IN ({$admins_query})
    ";

	$results = $wpdb->get_results( $admins_without_mfa_query );

	if ( $results ) {
		$usernames = [];

		foreach ( $results as $result ) {
			$usernames[] = $result->user_login;
		}

		$response = [
			'id'          => 999,
			'status'      => 'new',
			'type'        => 'wfAdminWithoutMFA',
			'severity'    => 75,
			'shortMsg'    => 'The administrator(s) without MFA was found.',
			'longMsg'     => 'Such administrator(s) is active, but without MFA configuration: ' . implode( ', ', $usernames ),
		];

		return (object) $response;
	}

	return null;
}

function wordfence_table_exists( $table_name ) {
	global $wpdb;

	$query = $wpdb->prepare(
		'SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = %s AND table_name = %s LIMIT 1',
		DB_NAME, $table_name
	);

	return $wpdb->get_var( $query ) > 0;
}
